<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Installment;
use App\Models\Customer;
use Illuminate\Http\Request;

class PaymentController extends Controller
{
    public function index(Request $request)
    {
        $query = Payment::with(['customer', 'installment.property']);

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('customer_id')) {
            $query->where('customer_id', $request->customer_id);
        }

        $payments = $query->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('payments.index', compact('payments'));
    }

    public function create(Request $request)
    {
        $installment = null;
        $customer = null;

        if ($request->has('installment_id')) {
            $installment = Installment::with(['customer', 'property'])->find($request->installment_id);
            $customer = $installment->customer;
        }
        if ($request->has('customer_id') && !$customer) {
        $customer = Customer::find($request->customer_id);
    }

        $customers = Customer::where('status', 'active')->get();

        return view('payments.create', compact('installment', 'customer', 'customers'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'installment_id' => 'required|exists:installments,id',
            'customer_id' => 'required|exists:customers,id',
            'amount' => 'required|numeric|min:0',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer,cheque,online',
            'reference_number' => 'nullable|string|max:255',
            'remarks' => 'nullable|string',
            'status' => 'required|in:pending,completed,failed,refunded',
        ]);

        // Check if amount exceeds remaining amount
        $installment = Installment::find($validated['installment_id']);
        $remaining = $installment->amount - $installment->paid_amount;

        if ($validated['amount'] > $remaining) {
            return back()->withErrors(['amount' => "Amount cannot exceed remaining amount: $remaining"])->withInput();
        }

        Payment::create($validated);

        // Update installment status
        if ($validated['status'] === 'completed') {
            $installment->paid_amount += $validated['amount'];
            $installment->updateStatus();
        }
        if ($request->has('add_another')) {
        return redirect()->route('payments.create')
            ->with('success', 'Payment recorded successfully. Add another?');
    }

        return redirect()->route('payments.index')
            ->with('success', 'Payment recorded successfully.');
    }

    public function show(Payment $payment)
    {
        $payment->load(['customer', 'installment.property']);

        return view('payments.show', compact('payment'));
    }

    public function edit(Payment $payment)
    {
        return view('payments.edit', compact('payment'));
    }

    public function update(Request $request, Payment $payment)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer,cheque,online',
            'reference_number' => 'nullable|string|max:255',
            'remarks' => 'nullable|string',
            'status' => 'required|in:pending,completed,failed,refunded',
        ]);

        $oldStatus = $payment->status;
        $oldAmount = $payment->amount;

        $payment->update($validated);

        // Update installment if status changed
        if ($oldStatus !== $validated['status'] || $oldAmount !== $validated['amount']) {
            $installment = $payment->installment;

            if ($oldStatus === 'completed') {
                $installment->paid_amount -= $oldAmount;
            }

            if ($validated['status'] === 'completed') {
                $installment->paid_amount += $validated['amount'];
            }

            $installment->updateStatus();
        }

        return redirect()->route('payments.show', $payment)
            ->with('success', 'Payment updated successfully.');
    }

    public function destroy(Payment $payment)
    {
        // Update installment before deleting payment
        if ($payment->status === 'completed') {
            $installment = $payment->installment;
            $installment->paid_amount -= $payment->amount;
            $installment->updateStatus();
        }

        $payment->delete();

        return redirect()->route('payments.index')
            ->with('success', 'Payment deleted successfully.');
    }
}
