<?php

namespace App\Http\Controllers;

use App\Models\Property;
use App\Models\Customer;
use App\Models\Payment;
use App\Models\Installment;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function index()
    {
        // Get summary statistics
        $stats = [
            'total_properties' => Property::count(),
            'total_customers' => Customer::count(),
            'active_installments' => Installment::where('status', 'pending')->count(),
            'overdue_installments' => Installment::where('status', 'overdue')->count(),
            'total_revenue' => Payment::where('status', 'completed')->sum('amount'),
            'pending_payments' => Payment::where('status', 'pending')->sum('amount'),
        ];

        // Get monthly revenue for current year
        $monthlyRevenue = Payment::selectRaw('
                YEAR(payment_date) as year,
                MONTH(payment_date) as month,
                SUM(amount) as total
            ')
            ->where('status', 'completed')
            ->whereYear('payment_date', date('Y'))
            ->groupBy('year', 'month')
            ->orderBy('year', 'desc')
            ->orderBy('month', 'desc')
            ->get();

        // Get recent payments
        $recentPayments = Payment::with(['customer', 'installment.property'])
            ->where('status', 'completed')
            ->orderBy('payment_date', 'desc')
            ->take(10)
            ->get();

        return view('reports.index', compact('stats', 'monthlyRevenue', 'recentPayments'));
    }

    public function properties(Request $request)
    {
        $query = Property::withCount(['installments', 'customers'])
            ->withSum('installments', 'paid_amount');

        // Apply filters
        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }

        if ($request->has('location') && $request->location) {
            $query->where('location', 'like', '%' . $request->location . '%');
        }

        $properties = $query->orderBy('created_at', 'desc')->get();

        $totalProperties = $properties->count();
        $totalValue = $properties->sum('total_price');
        $totalPaid = $properties->sum('installments_sum_paid_amount');

        return view('reports.properties', compact('properties', 'totalProperties', 'totalValue', 'totalPaid'));
    }

    public function customers(Request $request)
    {
        $query = Customer::withCount(['installments', 'payments', 'properties'])
            ->withSum('payments', 'amount');

        // Apply filters
        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }

        $customers = $query->orderBy('created_at', 'desc')->get();

        // Calculate totals
        $totalCustomers = $customers->count();
        $totalPayments = $customers->sum('payments_sum_amount');
        $activeCustomers = $customers->where('status', 'active')->count();

        return view('reports.customers', compact('customers', 'totalCustomers', 'totalPayments', 'activeCustomers'));
    }

    public function payments(Request $request)
    {
        $query = Payment::with(['customer', 'installment.property']);

        // Apply date filters
        if ($request->has('start_date') && $request->start_date) {
            $query->whereDate('payment_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->whereDate('payment_date', '<=', $request->end_date);
        }

        // Apply status filter
        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }

        // Apply method filter
        if ($request->has('method') && $request->method != 'all') {
            $query->where('payment_method', $request->method);
        }

        $payments = $query->orderBy('payment_date', 'desc')->get();

        // Calculate totals
        $totalAmount = $payments->where('status', 'completed')->sum('amount');
        $totalCount = $payments->count();
        $completedCount = $payments->where('status', 'completed')->count();

        return view('reports.payments', compact('payments', 'totalAmount', 'totalCount', 'completedCount'));
    }

    public function installments(Request $request)
    {
        $query = Installment::with(['property', 'customer']);

        // Apply status filter
        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }

        // Apply date filters
        if ($request->has('start_date') && $request->start_date) {
            $query->whereDate('due_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->whereDate('due_date', '<=', $request->end_date);
        }

        // Apply property filter
        if ($request->has('property_id') && $request->property_id) {
            $query->where('property_id', $request->property_id);
        }

        $installments = $query->orderBy('due_date', 'asc')->get();

        // Calculate totals
        $totalAmount = $installments->sum('amount');
        $totalPaid = $installments->sum('paid_amount');
        $pendingCount = $installments->where('status', 'pending')->count();
        $overdueCount = $installments->where('status', 'overdue')->count();

        $properties = Property::all();

        return view('reports.installments', compact('installments', 'totalAmount', 'totalPaid', 'pendingCount', 'overdueCount', 'properties'));
    }

    public function generate(Request $request)
    {
        $type = $request->type;
        $startDate = $request->start_date;
        $endDate = $request->end_date;

        switch ($type) {
            case 'daily':
                $payments = Payment::whereDate('payment_date', Carbon::today())
                    ->where('status', 'completed')
                    ->with(['customer', 'installment.property'])
                    ->get();
                $title = "Daily Payment Report - " . Carbon::today()->format('d M Y');
                break;

            case 'weekly':
                $payments = Payment::whereBetween('payment_date',
                    [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()])
                    ->where('status', 'completed')
                    ->with(['customer', 'installment.property'])
                    ->get();
                $title = "Weekly Payment Report - " . Carbon::now()->startOfWeek()->format('d M Y') . " to " . Carbon::now()->endOfWeek()->format('d M Y');
                break;

            case 'monthly':
                $payments = Payment::whereMonth('payment_date', Carbon::now()->month)
                    ->whereYear('payment_date', Carbon::now()->year)
                    ->where('status', 'completed')
                    ->with(['customer', 'installment.property'])
                    ->get();
                $title = "Monthly Payment Report - " . Carbon::now()->format('F Y');
                break;

            case 'custom':
                $payments = Payment::whereBetween('payment_date', [$startDate, $endDate])
                    ->where('status', 'completed')
                    ->with(['customer', 'installment.property'])
                    ->get();
                $title = "Custom Payment Report - " . Carbon::parse($startDate)->format('d M Y') . " to " . Carbon::parse($endDate)->format('d M Y');
                break;

            default:
                $payments = collect();
                $title = "Report";
        }

        $totalAmount = $payments->sum('amount');
        $totalCount = $payments->count();

        return view('reports.generate', compact('payments', 'title', 'totalAmount', 'totalCount'));
    }

    public function dashboard()
    {
        // Get today's payments
        $todayPayments = Payment::whereDate('payment_date', Carbon::today())
            ->where('status', 'completed')
            ->sum('amount');

        // Get week's payments
        $weekPayments = Payment::whereBetween('payment_date',
            [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()])
            ->where('status', 'completed')
            ->sum('amount');

        // Get month's payments
        $monthPayments = Payment::whereMonth('payment_date', Carbon::now()->month)
            ->whereYear('payment_date', Carbon::now()->year)
            ->where('status', 'completed')
            ->sum('amount');

        // Get overdue installments
        $overdueInstallments = Installment::where('status', 'overdue')
            ->with(['property', 'customer'])
            ->orderBy('due_date', 'asc')
            ->take(10)
            ->get();

        // Get upcoming installments (next 7 days)
        $upcomingInstallments = Installment::where('status', 'pending')
            ->whereBetween('due_date', [Carbon::today(), Carbon::today()->addDays(7)])
            ->with(['property', 'customer'])
            ->orderBy('due_date', 'asc')
            ->take(10)
            ->get();

        // Get payment methods distribution
        $paymentMethods = Payment::select('payment_method', DB::raw('SUM(amount) as total'))
            ->where('status', 'completed')
            ->groupBy('payment_method')
            ->get();

        // Get monthly revenue for chart
        $monthlyRevenue = Payment::selectRaw('
                MONTH(payment_date) as month,
                SUM(amount) as total
            ')
            ->where('status', 'completed')
            ->whereYear('payment_date', date('Y'))
            ->groupBy('month')
            ->orderBy('month', 'asc')
            ->get();

        return view('reports.dashboard', compact(
            'todayPayments',
            'weekPayments',
            'monthPayments',
            'overdueInstallments',
            'upcomingInstallments',
            'paymentMethods',
            'monthlyRevenue'
        ));
    }
}
