<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Property extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'location',
        'total_price',
        'down_payment',
        'installment_years',
        'installment_amount',
        'total_installments',
        'status',
        'description',
    ];

    protected $casts = [
        'total_price' => 'decimal:2',
        'down_payment' => 'decimal:2',
        'installment_amount' => 'decimal:2',
    ];

    public function installments()
    {
        return $this->hasMany(Installment::class);
    }

    public function customers()
    {
        return $this->belongsToMany(Customer::class, 'installments')
                    ->withPivot('installment_number', 'status', 'amount')
                    ->withTimestamps();
        return $this->belongsToMany(Customer::class, 'customer_property')
        ->withPivot(
            'purchase_date',
            'total_amount',
            'down_payment',
            'installment_amount',
            'total_installments',
            'payment_frequency',
            'notes')
             ->withTimestamps();
    }

    public function getRemainingAmountAttribute()
    {
        $paid = $this->installments()->where('status', 'paid')->sum('paid_amount');
        return $this->total_price - $this->down_payment - $paid;
    }

    public function getProgressAttribute()
    {
        $total = $this->total_installments;
        $completed = $this->installments()->where('status', 'paid')->count();
        return $total > 0 ? ($completed / $total) * 100 : 0;
    }

}
