@extends('layouts.app')

@section('title', 'Add New Customer')
@section('page-title', 'Add New Customer')

@section('content')
<div class="row justify-content-center">
    <div class="col-md-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Customer Information</h5>
            </div>
            <div class="card-body">
                <form method="POST" action="{{ route('customers.store') }}" id="customerForm" enctype="multipart/form-data">
                    @csrf

                    <div class="row">
                        <!-- Basic Information -->
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="name" class="form-label">Full Name *</label>
                                <input type="text" class="form-control @error('name') is-invalid @enderror"
                                       id="name" name="name" value="{{ old('name') }}"
                                       placeholder="Enter customer full name" required>
                                @error('name')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="email" class="form-label">Email Address *</label>
                                <input type="email" class="form-control @error('email') is-invalid @enderror"
                                       id="email" name="email" value="{{ old('email') }}"
                                       placeholder="Enter email address" required>
                                @error('email')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="phone" class="form-label">Phone Number *</label>
                                <div class="input-group">
                                    <span class="input-group-text">+92</span>
                                    <input type="text" class="form-control @error('phone') is-invalid @enderror"
                                           id="phone" name="phone" value="{{ old('phone') }}"
                                           placeholder="3001234567" pattern="[0-9]{10}" required>
                                </div>
                                <div class="form-text">Format: 3001234567 (without dashes)</div>
                                @error('phone')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="cnic" class="form-label">CNIC Number *</label>
                                <input type="text" class="form-control @error('cnic') is-invalid @enderror"
                                       id="cnic" name="cnic" value="{{ old('cnic') }}"
                                       placeholder="12345-1234567-1" pattern="[0-9]{5}-[0-9]{7}-[0-9]{1}" required>
                                <div class="form-text">Format: 12345-1234567-1</div>
                                @error('cnic')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="date_of_birth" class="form-label">Date of Birth</label>
                                <input type="date" class="form-control @error('date_of_birth') is-invalid @enderror"
                                       id="date_of_birth" name="date_of_birth" value="{{ old('date_of_birth') }}">
                                @error('date_of_birth')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="status" class="form-label">Status *</label>
                                <select class="form-control @error('status') is-invalid @enderror"
                                        id="status" name="status" required>
                                    <option value="">Select Status</option>
                                    <option value="active" {{ old('status') == 'active' ? 'selected' : '' }}>Active</option>
                                    <option value="inactive" {{ old('status') == 'inactive' ? 'selected' : '' }}>Inactive</option>
                                    <option value="blacklisted" {{ old('status') == 'blacklisted' ? 'selected' : '' }}>Blacklisted</option>
                                </select>
                                @error('status')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="address" class="form-label">Complete Address *</label>
                        <textarea class="form-control @error('address') is-invalid @enderror"
                                  id="address" name="address" rows="3"
                                  placeholder="Enter complete address" required>{{ old('address') }}</textarea>
                        @error('address')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <!-- Document Upload Section -->
                    <div class="card mt-4">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0">Document Uploads</h5>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <!-- CNIC Front -->
                                <div class="col-md-3">
                                    <div class="mb-3">
                                        <label for="cnic_front" class="form-label">CNIC Front Side</label>
                                        <div class="custom-file">
                                            <input type="file" class="custom-file-input @error('cnic_front') is-invalid @enderror"
                                                   id="cnic_front" name="cnic_front">
                                            <label class="custom-file-label" for="cnic_front">Choose file</label>
                                        </div>
                                        <small class="form-text text-muted">Max: 2MB (JPG, PNG, GIF)</small>
                                        @error('cnic_front')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>

                                <!-- CNIC Back -->
                                <div class="col-md-3">
                                    <div class="mb-3">
                                        <label for="cnic_back" class="form-label">CNIC Back Side</label>
                                        <div class="custom-file">
                                            <input type="file" class="custom-file-input @error('cnic_back') is-invalid @enderror"
                                                   id="cnic_back" name="cnic_back">
                                            <label class="custom-file-label" for="cnic_back">Choose file</label>
                                        </div>
                                        <small class="form-text text-muted">Max: 2MB (JPG, PNG, GIF)</small>
                                        @error('cnic_back')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>

                                <!-- Customer Picture -->
                                <div class="col-md-3">
                                    <div class="mb-3">
                                        <label for="customer_picture" class="form-label">Customer Picture</label>
                                        <div class="custom-file">
                                            <input type="file" class="custom-file-input @error('customer_picture') is-invalid @enderror"
                                                   id="customer_picture" name="customer_picture">
                                            <label class="custom-file-label" for="customer_picture">Choose file</label>
                                        </div>
                                        <small class="form-text text-muted">Max: 2MB (JPG, PNG, GIF)</small>
                                        @error('customer_picture')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>

                                <!-- Other Documents -->
                                <div class="col-md-3">
                                    <div class="mb-3">
                                        <label for="other_documents" class="form-label">Other Documents</label>
                                        <div class="custom-file">
                                            <input type="file" class="custom-file-input @error('other_documents.*') is-invalid @enderror"
                                                   id="other_documents" name="other_documents[]" multiple>
                                            <label class="custom-file-label" for="other_documents">Choose files</label>
                                        </div>
                                        <small class="form-text text-muted">Max: 5MB each (PDF, DOC, JPG, PNG)</small>
                                        @error('other_documents.*')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>
                            <div class="alert alert-info mt-2">
                                <i class="fas fa-info-circle me-2"></i>
                                Documents will be stored securely. You can upload them later from the customer profile if needed.
                            </div>
                        </div>
                    </div>

                    <div class="alert alert-info mt-4">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Note:</strong> All fields marked with * are required. You can add documents and granters after creating the customer profile.
                    </div>

                    <div class="mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i> Create Customer
                        </button>
                        <button type="button" class="btn btn-outline-primary" onclick="saveAndAddAnother()">
                            <i class="fas fa-plus-circle me-2"></i> Save & Add Another
                        </button>
                        <a href="{{ route('customers.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times me-2"></i> Cancel
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
    $(document).ready(function() {
        // Format phone number
        $('#phone').on('input', function() {
            let value = $(this).val().replace(/\D/g, '');
            if (value.length > 10) value = value.substring(0, 10);
            $(this).val(value);
        });

        // Format CNIC
        $('#cnic').on('input', function() {
            let value = $(this).val().replace(/\D/g, '');
            if (value.length > 13) value = value.substring(0, 13);

            // Add dashes
            if (value.length > 5) {
                value = value.substring(0, 5) + '-' + value.substring(5);
            }
            if (value.length > 13) {
                value = value.substring(0, 13) + '-' + value.substring(13);
            }
            $(this).val(value);
        });

        // File input label updates
        document.querySelectorAll('.custom-file-input').forEach(function(input) {
            input.addEventListener('change', function(e) {
                var fileName = '';
                if (this.files && this.files.length > 1) {
                    fileName = this.files.length + ' files selected';
                } else if (this.files && this.files.length === 1) {
                    fileName = e.target.value.split('\\').pop();
                }
                if (fileName) {
                    $(this).next('.custom-file-label').html(fileName);
                }
            });
        });

        // Form validation
        $('#customerForm').validate({
            rules: {
                name: 'required',
                email: {
                    required: true,
                    email: true
                },
                phone: {
                    required: true,
                    minlength: 10,
                    maxlength: 10,
                    digits: true
                },
                cnic: {
                    required: true,
                    pattern: /^[0-9]{5}-[0-9]{7}-[0-9]{1}$/
                },
                address: 'required',
                status: 'required',
                cnic_front: {
                    extension: "jpg|jpeg|png|gif",
                    filesize: 2048
                },
                cnic_back: {
                    extension: "jpg|jpeg|png|gif",
                    filesize: 2048
                },
                customer_picture: {
                    extension: "jpg|jpeg|png|gif",
                    filesize: 2048
                },
                'other_documents[]': {
                    extension: "pdf|jpg|jpeg|png|doc|docx",
                    filesize: 5120
                }
            },
            messages: {
                name: 'Please enter customer name',
                email: {
                    required: 'Please enter email address',
                    email: 'Please enter a valid email address'
                },
                phone: {
                    required: 'Please enter phone number',
                    minlength: 'Phone number must be 10 digits',
                    maxlength: 'Phone number must be 10 digits',
                    digits: 'Please enter numbers only'
                },
                cnic: {
                    required: 'Please enter CNIC number',
                    pattern: 'CNIC format must be 12345-1234567-1'
                },
                address: 'Please enter address',
                status: 'Please select status'
            },
            errorElement: 'div',
            errorClass: 'invalid-feedback',
            highlight: function(element) {
                $(element).addClass('is-invalid');
            },
            unhighlight: function(element) {
                $(element).removeClass('is-invalid');
            }
        });
    });

    function saveAndAddAnother() {
        $('#customerForm').append('<input type="hidden" name="add_another" value="1">');
        $('#customerForm').submit();
    }
</script>
@endpush
@endsection
