@extends('layouts.app')

@section('title', 'Installments')
@section('page-title', 'Installment Management')

@section('content')
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">All Installments</h5>
        <div>
            <a href="{{ route('properties.create') }}" class="btn btn-primary">
                <i class="fas fa-plus me-2"></i> Create Property with Installments
            </a>
        </div>
    </div>

    <div class="card-body">
        <!-- Filters -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="input-group">
                    <span class="input-group-text"><i class="fas fa-search"></i></span>
                    <input type="text" id="searchInput" class="form-control" placeholder="Search...">
                </div>
            </div>
            <div class="col-md-3">
                <select id="statusFilter" class="form-control">
                    <option value="">All Status</option>
                    <option value="paid">Paid</option>
                    <option value="pending">Pending</option>
                    <option value="overdue">Overdue</option>
                    <option value="partial">Partial</option>
                </select>
            </div>
            <div class="col-md-3">
                <select id="propertyFilter" class="form-control">
                    <option value="">All Properties</option>
                    @foreach(\App\Models\Property::all() as $property)
                        <option value="{{ $property->id }}">{{ $property->name }}</option>
                    @endforeach
                </select>
            </div>
            <div class="col-md-3">
                <input type="date" id="dateFilter" class="form-control" placeholder="Due Date">
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card bg-primary text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-0">Total Installments</h6>
                                <h3 class="mb-0">{{ $installments->total() }}</h3>
                            </div>
                            <i class="fas fa-calendar-alt fa-2x opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-success text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-0">Paid</h6>
                                <h3 class="mb-0">{{ $installments->where('status', 'paid')->count() }}</h3>
                            </div>
                            <i class="fas fa-check-circle fa-2x opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-warning text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-0">Pending</h6>
                                <h3 class="mb-0">{{ $installments->where('status', 'pending')->count() }}</h3>
                            </div>
                            <i class="fas fa-clock fa-2x opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-danger text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-0">Overdue</h6>
                                <h3 class="mb-0">{{ $installments->where('status', 'overdue')->count() }}</h3>
                            </div>
                            <i class="fas fa-exclamation-triangle fa-2x opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Installments Table -->
        <div class="table-responsive">
            <table class="table table-hover" id="installmentsTable">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Property</th>
                        <th>Customer</th>
                        <th>Due Date</th>
                        <th>Installment #</th>
                        <th>Amount</th>
                        <th>Paid</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($installments as $installment)
                    <tr class="installment-row"
                        data-status="{{ $installment->status }}"
                        data-property="{{ $installment->property_id }}"
                        data-date="{{ $installment->due_date->format('Y-m-d') }}">
                        <td>{{ $loop->iteration }}</td>
                        <td>
                            <strong>{{ $installment->property->name }}</strong>
                            <div class="small text-muted">{{ Str::limit($installment->property->location, 30) }}</div>
                        </td>
                        <td>
                            <div class="d-flex align-items-center">
                                <div class="avatar bg-primary text-white rounded-circle d-flex align-items-center justify-content-center me-2"
                                     style="width: 30px; height: 30px; font-size: 12px;">
                                    {{ substr($installment->customer->name, 0, 1) }}
                                </div>
                                <div>
                                    <strong>{{ $installment->customer->name }}</strong>
                                    <div class="small text-muted">{{ $installment->customer->phone }}</div>
                                </div>
                            </div>
                        </td>
                        <td>
                            {{ $installment->due_date->format('d M Y') }}
                            @if($installment->isOverdue())
                                <br><small class="text-danger">
                                    <i class="fas fa-exclamation-circle"></i> Overdue by
                                    {{ now()->diffInDays($installment->due_date) }} days
                                </small>
                            @endif
                        </td>
                        <td>
                            <span class="badge bg-secondary">#{{ $installment->installment_number }}</span>
                        </td>
                        <td>
                            <strong>Rs. {{ number_format($installment->amount, 2) }}</strong>
                        </td>
                        <td>
                            Rs. {{ number_format($installment->paid_amount, 2) }}
                            @if($installment->paid_amount > 0)
                                <div class="progress mt-1" style="height: 5px;">
                                    <div class="progress-bar bg-success"
                                         style="width: {{ ($installment->paid_amount / $installment->amount) * 100 }}%">
                                    </div>
                                </div>
                            @endif
                        </td>
                        <td>
                            @php
                                $statusColors = [
                                    'paid' => 'success',
                                    'pending' => 'warning',
                                    'overdue' => 'danger',
                                    'partial' => 'info'
                                ];
                                $statusIcons = [
                                    'paid' => 'check-circle',
                                    'pending' => 'clock',
                                    'overdue' => 'exclamation-triangle',
                                    'partial' => 'money-check-alt'
                                ];
                            @endphp
                            <span class="badge bg-{{ $statusColors[$installment->status] }}">
                                <i class="fas fa-{{ $statusIcons[$installment->status] }} me-1"></i>
                                {{ ucfirst($installment->status) }}
                            </span>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm" role="group">
                                @if($installment->status != 'paid')
                                    <a href="{{ route('payments.create', ['installment_id' => $installment->id]) }}"
                                       class="btn btn-success" title="Record Payment">
                                        <i class="fas fa-credit-card"></i>
                                    </a>
                                @endif
                                <a href="{{ route('installments.show', $installment) }}"
                                   class="btn btn-info" title="View Details">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <a href="{{ route('installments.edit', $installment) }}"
                                   class="btn btn-warning" title="Edit">
                                    <i class="fas fa-edit"></i>
                                </a>
                                @if($installment->status == 'pending')
                                    <button type="button" class="btn btn-primary"
                                            onclick="markAsPaid({{ $installment->id }})" title="Mark as Paid">
                                        <i class="fas fa-check"></i>
                                    </button>
                                @endif
                            </div>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="9" class="text-center py-5">
                            <i class="fas fa-calendar-alt fa-3x text-muted mb-3"></i>
                            <h5>No Installments Found</h5>
                            <p class="text-muted">Start by creating a property with installments</p>
                            <a href="{{ route('properties.create') }}" class="btn btn-primary">
                                <i class="fas fa-plus me-2"></i> Create Property
                            </a>
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        @if($installments->hasPages())
        <div class="d-flex justify-content-between align-items-center mt-3">
            <div>
                Showing {{ $installments->firstItem() }} to {{ $installments->lastItem() }} of {{ $installments->total() }} entries
            </div>
            <div>
                {{ $installments->links() }}
            </div>
        </div>
        @endif
    </div>
</div>

<!-- Mark as Paid Modal -->
<div class="modal fade" id="markPaidModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Mark Installment as Paid</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to mark this installment as paid?</p>
                <div id="installmentDetails"></div>
                <form id="markPaidForm" method="POST">
                    @csrf
                    <input type="hidden" name="installment_id" id="installment_id">
                    <div class="mb-3">
                        <label for="payment_date" class="form-label">Payment Date</label>
                        <input type="date" class="form-control" id="payment_date" name="payment_date"
                               value="{{ date('Y-m-d') }}" required>
                    </div>
                    <div class="mb-3">
                        <label for="payment_method" class="form-label">Payment Method</label>
                        <select class="form-control" id="payment_method" name="payment_method" required>
                            <option value="cash">Cash</option>
                            <option value="bank_transfer">Bank Transfer</option>
                            <option value="cheque">Cheque</option>
                            <option value="online">Online</option>
                        </select>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="submit" form="markPaidForm" class="btn btn-primary">Mark as Paid</button>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
    // Search functionality
    $('#searchInput').on('keyup', function() {
        const value = $(this).val().toLowerCase();
        $('#installmentsTable tbody tr').filter(function() {
            $(this).toggle($(this).text().toLowerCase().indexOf(value) > -1);
        });
    });

    // Filter by status
    $('#statusFilter').on('change', function() {
        const status = $(this).val();
        filterTable();
    });

    // Filter by property
    $('#propertyFilter').on('change', function() {
        filterTable();
    });

    // Filter by date
    $('#dateFilter').on('change', function() {
        filterTable();
    });

    function filterTable() {
        const status = $('#statusFilter').val();
        const property = $('#propertyFilter').val();
        const date = $('#dateFilter').val();

        $('#installmentsTable tbody tr').each(function() {
            const showStatus = !status || $(this).data('status') === status;
            const showProperty = !property || $(this).data('property') == property;
            const showDate = !date || $(this).data('date') === date;

            $(this).toggle(showStatus && showProperty && showDate);
        });
    }

    // Mark as Paid
    function markAsPaid(installmentId) {
        // Fetch installment details
        fetch(`/installments/${installmentId}`)
            .then(response => response.json())
            .then(data => {
                $('#installmentDetails').html(`
                    <div class="alert alert-info">
                        <strong>Property:</strong> ${data.property.name}<br>
                        <strong>Customer:</strong> ${data.customer.name}<br>
                        <strong>Installment #:</strong> ${data.installment_number}<br>
                        <strong>Amount:</strong> Rs. ${parseFloat(data.amount).toFixed(2)}
                    </div>
                `);
                $('#installment_id').val(installmentId);
                $('#markPaidForm').attr('action', `/installments/${installmentId}/mark-paid`);
                new bootstrap.Modal(document.getElementById('markPaidModal')).show();
            })
            .catch(error => console.error('Error:', error));
    }

    // Initialize DataTable
    $(document).ready(function() {
        $('#installmentsTable').DataTable({
            paging: false,
            searching: false,
            info: false,
            order: [[3, 'asc']]
        });
    });

    // Export to CSV
    function exportToCSV() {
        const rows = [];
        const headers = ['Property', 'Customer', 'Due Date', 'Installment #', 'Amount', 'Paid Amount', 'Status'];

        $('#installmentsTable tbody tr').each(function() {
            const row = [];
            row.push($(this).find('td:nth-child(2) strong').text().trim());
            row.push($(this).find('td:nth-child(3) strong').text().trim());
            row.push($(this).find('td:nth-child(4)').text().split('\n')[0].trim());
            row.push($(this).find('td:nth-child(5) span').text().replace('#', '').trim());
            row.push($(this).find('td:nth-child(6) strong').text().replace('Rs.', '').trim());
            row.push($(this).find('td:nth-child(7)').text().split('\n')[0].replace('Rs.', '').trim());
            row.push($(this).find('td:nth-child(8) span').text().trim());

            rows.push(row.join(','));
        });

        const csvContent = [headers.join(','), ...rows].join('\n');
        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `installments_${new Date().toISOString().split('T')[0]}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
    }
</script>
@endpush

@push('styles')
<style>
    .avatar {
        font-weight: bold;
    }

    .badge {
        min-width: 90px;
        text-align: center;
    }

    .installment-row:hover {
        background-color: rgba(67, 97, 238, 0.05);
    }

    .progress {
        border-radius: 10px;
    }

    .card.bg-primary,
    .card.bg-success,
    .card.bg-warning,
    .card.bg-danger {
        transition: transform 0.3s;
    }

    .card.bg-primary:hover,
    .card.bg-success:hover,
    .card.bg-warning:hover,
    .card.bg-danger:hover {
        transform: translateY(-5px);
    }
</style>
@endpush
@endsection
