@extends('layouts.app')

@section('title', 'Record New Payment')
@section('page-title', 'Record New Payment')

@section('content')
<div class="row justify-content-center">
    <div class="col-md-8">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Payment Information</h5>
            </div>
            <div class="card-body">
                <form method="POST" action="{{ route('payments.store') }}" id="paymentForm">
                    @csrf

                    <div class="row">
                        <!-- Customer Selection -->
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="customer_id" class="form-label">Customer *</label>
                                <select class="form-control @error('customer_id') is-invalid @enderror"
                                        id="customer_id" name="customer_id" required
                                        onchange="loadCustomerInstallments(this.value)">
                                    <option value="">Select a customer</option>
                                    @foreach($customers as $customer)
                                        <option value="{{ $customer->id }}"
                                                {{ old('customer_id', $customer->id == ($customer ?? null)?->id ? $customer->id : '') == $customer->id ? 'selected' : '' }}>
                                            {{ $customer->name }} - {{ $customer->phone }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('customer_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <!-- Installment Selection -->
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="installment_id" class="form-label">Installment *</label>
                                <select class="form-control @error('installment_id') is-invalid @enderror"
                                        id="installment_id" name="installment_id" required
                                        onchange="loadInstallmentDetails(this.value)">
                                    <option value="">Select an installment</option>
                                    @if(isset($installment) && $installment)
                                        <option value="{{ $installment->id }}" selected>
                                            Property: {{ $installment->property->name }} |
                                            Installment #{{ $installment->installment_number }} |
                                            Due: {{ $installment->due_date->format('d/m/Y') }}
                                        </option>
                                    @endif
                                </select>
                                @error('installment_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <!-- Installment Details -->
                    <div id="installmentDetails" class="card bg-light mb-4" style="display: none;">
                        <div class="card-body">
                            <h6 class="card-title">Installment Details</h6>
                            <div class="row">
                                <div class="col-md-4">
                                    <small class="text-muted">Property</small>
                                    <p class="mb-1"><strong id="propertyName">-</strong></p>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted">Total Amount</small>
                                    <p class="mb-1"><strong id="installmentAmount">Rs. 0.00</strong></p>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted">Paid Amount</small>
                                    <p class="mb-1"><strong id="paidAmount">Rs. 0.00</strong></p>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted">Remaining Amount</small>
                                    <p class="mb-1"><strong id="remainingAmount" class="text-warning">Rs. 0.00</strong></p>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted">Due Date</small>
                                    <p class="mb-1"><strong id="dueDate">-</strong></p>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted">Status</small>
                                    <p class="mb-1">
                                        <span id="installmentStatus" class="badge bg-secondary">-</span>
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <!-- Payment Details -->
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="amount" class="form-label">Payment Amount (PKR) *</label>
                                <div class="input-group">
                                    <span class="input-group-text">Rs.</span>
                                    <input type="number" step="0.01" class="form-control @error('amount') is-invalid @enderror"
                                           id="amount" name="amount" value="{{ old('amount', isset($installment) ? $installment->remaining_amount : '') }}"
                                           placeholder="Enter payment amount" required min="0">
                                </div>
                                @error('amount')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="payment_date" class="form-label">Payment Date *</label>
                                <input type="date" class="form-control @error('payment_date') is-invalid @enderror"
                                       id="payment_date" name="payment_date"
                                       value="{{ old('payment_date', date('Y-m-d')) }}" required>
                                @error('payment_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="payment_method" class="form-label">Payment Method *</label>
                                <select class="form-control @error('payment_method') is-invalid @enderror"
                                        id="payment_method" name="payment_method" required>
                                    <option value="">Select method</option>
                                    <option value="cash" {{ old('payment_method') == 'cash' ? 'selected' : '' }}>Cash</option>
                                    <option value="bank_transfer" {{ old('payment_method') == 'bank_transfer' ? 'selected' : '' }}>Bank Transfer</option>
                                    <option value="cheque" {{ old('payment_method') == 'cheque' ? 'selected' : '' }}>Cheque</option>
                                    <option value="online" {{ old('payment_method') == 'online' ? 'selected' : '' }}>Online</option>
                                </select>
                                @error('payment_method')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="reference_number" class="form-label">Reference Number</label>
                                <input type="text" class="form-control @error('reference_number') is-invalid @enderror"
                                       id="reference_number" name="reference_number"
                                       value="{{ old('reference_number') }}"
                                       placeholder="Enter reference/transaction number">
                                <div class="form-text">Required for bank transfer, cheque, and online payments</div>
                                @error('reference_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="status" class="form-label">Payment Status *</label>
                                <select class="form-control @error('status') is-invalid @enderror"
                                        id="status" name="status" required>
                                    <option value="pending" {{ old('status') == 'pending' ? 'selected' : '' }}>Pending</option>
                                    <option value="completed" {{ old('status') == 'completed' ? 'selected' : '' }}>Completed</option>
                                    <option value="failed" {{ old('status') == 'failed' ? 'selected' : '' }}>Failed</option>
                                </select>
                                @error('status')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="remarks" class="form-label">Remarks</label>
                        <textarea class="form-control @error('remarks') is-invalid @enderror"
                                  id="remarks" name="remarks" rows="3"
                                  placeholder="Enter any remarks about this payment">{{ old('remarks') }}</textarea>
                        @error('remarks')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Note:</strong> After recording the payment, the installment status will be updated automatically based on the paid amount.
                    </div>

                    <div class="mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i> Record Payment
                        </button>
                        <button type="button" class="btn btn-outline-primary" onclick="saveAndAddAnother()">
                            <i class="fas fa-plus-circle me-2"></i> Save & Add Another
                        </button>
                        <a href="{{ route('payments.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times me-2"></i> Cancel
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
    $(document).ready(function() {
        // Initialize with customer if provided
        @if(isset($customer) && $customer)
            $('#customer_id').val('{{ $customer->id }}').trigger('change');
        @endif

        // Initialize with installment if provided
        @if(isset($installment) && $installment)
            loadInstallmentDetails({{ $installment->id }});
        @endif

        // Load customer installments
        function loadCustomerInstallments(customerId) {
            if (!customerId) {
                $('#installment_id').html('<option value="">Select an installment</option>');
                return;
            }

            $('#installment_id').html('<option value="">Loading installments...</option>');

            fetch(`/api/customers/${customerId}/installments`)
                .then(response => response.json())
                .then(data => {
                    if (data.length === 0) {
                        $('#installment_id').html('<option value="">No installments found</option>');
                        return;
                    }

                    let options = '<option value="">Select an installment</option>';
                    data.forEach(installment => {
                        options += `<option value="${installment.id}">
                            Property: ${installment.property_name} |
                            Installment #${installment.installment_number} |
                            Due: ${installment.due_date} |
                            Remaining: Rs. ${parseFloat(installment.remaining_amount).toFixed(2)}
                        </option>`;
                    });

                    $('#installment_id').html(options);
                })
                .catch(error => {
                    console.error('Error:', error);
                    $('#installment_id').html('<option value="">Error loading installments</option>');
                });
        }

        // Load installment details
        function loadInstallmentDetails(installmentId) {
            if (!installmentId) {
                $('#installmentDetails').hide();
                return;
            }

            fetch(`/api/installments/${installmentId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.error) {
                        alert(data.error);
                        return;
                    }

                    // Update installment details
                    $('#propertyName').text(data.property.name);
                    $('#installmentAmount').text('Rs. ' + parseFloat(data.amount).toFixed(2));
                    $('#paidAmount').text('Rs. ' + parseFloat(data.paid_amount).toFixed(2));
                    $('#remainingAmount').text('Rs. ' + parseFloat(data.remaining_amount).toFixed(2));
                    $('#dueDate').text(data.due_date);

                    // Update status badge
                    const statusColors = {
                        'paid': 'success',
                        'pending': 'warning',
                        'overdue': 'danger',
                        'partial': 'info'
                    };
                    $('#installmentStatus').removeClass().addClass('badge bg-' + statusColors[data.status]);
                    $('#installmentStatus').text(data.status.charAt(0).toUpperCase() + data.status.slice(1));

                    // Set payment amount to remaining amount
                    $('#amount').val(parseFloat(data.remaining_amount).toFixed(2));

                    // Show details card
                    $('#installmentDetails').show();
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading installment details');
                });
        }

        // Validate payment amount doesn't exceed remaining amount
        $('#amount').on('change', function() {
            const remainingAmount = parseFloat($('#remainingAmount').text().replace('Rs.', '')) || 0;
            const paymentAmount = parseFloat($(this).val()) || 0;

            if (paymentAmount > remainingAmount) {
                alert('Payment amount cannot exceed remaining amount: Rs. ' + remainingAmount.toFixed(2));
                $(this).val(remainingAmount.toFixed(2));
            }
        });

        // Validate reference number based on payment method
        $('#payment_method').on('change', function() {
            const method = $(this).val();
            const referenceField = $('#reference_number');

            if (method === 'cash') {
                referenceField.removeAttr('required');
            } else {
                referenceField.attr('required', 'required');
            }
        });

        // Form validation
        $('#paymentForm').validate({
            rules: {
                customer_id: 'required',
                installment_id: 'required',
                amount: {
                    required: true,
                    min: 1,
                    max: function() {
                        const remaining = parseFloat($('#remainingAmount').text().replace('Rs.', '')) || 0;
                        return remaining;
                    }
                },
                payment_date: 'required',
                payment_method: 'required',
                reference_number: {
                    required: function() {
                        const method = $('#payment_method').val();
                        return method !== 'cash' && method !== '';
                    }
                },
                status: 'required'
            },
            messages: {
                customer_id: 'Please select a customer',
                installment_id: 'Please select an installment',
                amount: {
                    required: 'Please enter payment amount',
                    min: 'Amount must be at least Rs. 1',
                    max: 'Amount cannot exceed remaining amount'
                },
                payment_date: 'Please select payment date',
                payment_method: 'Please select payment method',
                reference_number: 'Reference number is required for this payment method',
                status: 'Please select payment status'
            },
            errorElement: 'div',
            errorClass: 'invalid-feedback',
            highlight: function(element) {
                $(element).addClass('is-invalid');
            },
            unhighlight: function(element) {
                $(element).removeClass('is-invalid');
            }
        });
    });

    function saveAndAddAnother() {
        $('#paymentForm').append('<input type="hidden" name="add_another" value="1">');
        $('#paymentForm').submit();
    }
</script>
@endpush
@endsection
