@extends('layouts.app')

@section('title', 'Edit Payment')
@section('page-title', 'Edit Payment')

@section('content')
<div class="row justify-content-center">
    <div class="col-md-8">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Edit Payment Information</h5>
            </div>
            <div class="card-body">
                <form method="POST" action="{{ route('payments.update', $payment) }}" id="paymentForm">
                    @csrf
                    @method('PUT')

                    <div class="row">
                        <!-- Payment Details -->
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="amount" class="form-label">Payment Amount (PKR) *</label>
                                <div class="input-group">
                                    <span class="input-group-text">Rs.</span>
                                    <input type="number" step="0.01" class="form-control @error('amount') is-invalid @enderror"
                                           id="amount" name="amount" value="{{ old('amount', $payment->amount) }}"
                                           placeholder="Enter payment amount" required min="0">
                                </div>
                                <div class="form-text">
                                    Installment Amount: Rs. {{ number_format($payment->installment->amount, 2) }}
                                </div>
                                @error('amount')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="payment_date" class="form-label">Payment Date *</label>
                                <input type="date" class="form-control @error('payment_date') is-invalid @enderror"
                                       id="payment_date" name="payment_date"
                                       value="{{ old('payment_date', $payment->payment_date->format('Y-m-d')) }}" required>
                                @error('payment_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="payment_method" class="form-label">Payment Method *</label>
                                <select class="form-control @error('payment_method') is-invalid @enderror"
                                        id="payment_method" name="payment_method" required>
                                    <option value="cash" {{ old('payment_method', $payment->payment_method) == 'cash' ? 'selected' : '' }}>Cash</option>
                                    <option value="bank_transfer" {{ old('payment_method', $payment->payment_method) == 'bank_transfer' ? 'selected' : '' }}>Bank Transfer</option>
                                    <option value="cheque" {{ old('payment_method', $payment->payment_method) == 'cheque' ? 'selected' : '' }}>Cheque</option>
                                    <option value="online" {{ old('payment_method', $payment->payment_method) == 'online' ? 'selected' : '' }}>Online</option>
                                </select>
                                @error('payment_method')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="reference_number" class="form-label">Reference Number</label>
                                <input type="text" class="form-control @error('reference_number') is-invalid @enderror"
                                       id="reference_number" name="reference_number"
                                       value="{{ old('reference_number', $payment->reference_number) }}"
                                       placeholder="Enter reference/transaction number">
                                <div class="form-text">Required for bank transfer, cheque, and online payments</div>
                                @error('reference_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="status" class="form-label">Payment Status *</label>
                                <select class="form-control @error('status') is-invalid @enderror"
                                        id="status" name="status" required>
                                    <option value="pending" {{ old('status', $payment->status) == 'pending' ? 'selected' : '' }}>Pending</option>
                                    <option value="completed" {{ old('status', $payment->status) == 'completed' ? 'selected' : '' }}>Completed</option>
                                    <option value="failed" {{ old('status', $payment->status) == 'failed' ? 'selected' : '' }}>Failed</option>
                                    <option value="refunded" {{ old('status', $payment->status) == 'refunded' ? 'selected' : '' }}>Refunded</option>
                                </select>
                                @error('status')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="remarks" class="form-label">Remarks</label>
                        <textarea class="form-control @error('remarks') is-invalid @enderror"
                                  id="remarks" name="remarks" rows="3"
                                  placeholder="Enter any remarks about this payment">{{ old('remarks', $payment->remarks) }}</textarea>
                        @error('remarks')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <!-- Payment Information Summary -->
                    <div class="card bg-light mb-4">
                        <div class="card-body">
                            <h6 class="card-title">Payment Information</h6>
                            <div class="row">
                                <div class="col-md-4">
                                    <small class="text-muted">Transaction ID</small>
                                    <p class="mb-1"><code>{{ $payment->transaction_id }}</code></p>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted">Customer</small>
                                    <p class="mb-1"><strong>{{ $payment->customer->name }}</strong></p>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted">Property</small>
                                    <p class="mb-1"><strong>{{ $payment->installment->property->name }}</strong></p>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted">Installment</small>
                                    <p class="mb-1"><strong>#{{ $payment->installment->installment_number }}</strong></p>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted">Current Status</small>
                                    <p class="mb-1">
                                        @php
                                            $statusColors = [
                                                'completed' => 'success',
                                                'pending' => 'warning',
                                                'failed' => 'danger',
                                                'refunded' => 'info'
                                            ];
                                        @endphp
                                        <span class="badge bg-{{ $statusColors[$payment->status] ?? 'secondary' }}">
                                            {{ ucfirst($payment->status) }}
                                        </span>
                                    </p>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted">Created</small>
                                    <p class="mb-1">{{ $payment->created_at->format('d M Y, h:i A') }}</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Installment Summary -->
                    <div class="card bg-light mb-4">
                        <div class="card-body">
                            <h6 class="card-title">Installment Summary</h6>
                            <div class="row">
                                <div class="col-md-3">
                                    <small class="text-muted">Installment Amount</small>
                                    <p class="mb-1"><strong>Rs. {{ number_format($payment->installment->amount, 2) }}</strong></p>
                                </div>
                                <div class="col-md-3">
                                    <small class="text-muted">Previously Paid</small>
                                    <p class="mb-1"><strong>Rs. {{ number_format($payment->installment->paid_amount - $payment->amount, 2) }}</strong></p>
                                </div>
                                <div class="col-md-3">
                                    <small class="text-muted">This Payment</small>
                                    <p class="mb-1"><strong>Rs. {{ number_format($payment->amount, 2) }}</strong></p>
                                </div>
                                <div class="col-md-3">
                                    <small class="text-muted">New Total Paid</small>
                                    <p class="mb-1">
                                        <strong id="newTotalPaid">
                                            Rs. {{ number_format($payment->installment->paid_amount, 2) }}
                                        </strong>
                                    </p>
                                </div>
                            </div>
                            <div class="row mt-2">
                                <div class="col-md-12">
                                    <small class="text-muted">Progress</small>
                                    <div class="progress" style="height: 15px;">
                                        <div class="progress-bar bg-success"
                                             style="width: {{ ($payment->installment->paid_amount / $payment->installment->amount) * 100 }}%;"
                                             id="progressBar">
                                        </div>
                                    </div>
                                    <div class="d-flex justify-content-between mt-1">
                                        <small class="text-muted">Rs. 0</small>
                                        <small class="text-muted">
                                            <span id="progressPercent">
                                                {{ round(($payment->installment->paid_amount / $payment->installment->amount) * 100) }}%
                                            </span>
                                        </small>
                                        <small class="text-muted">Rs. {{ number_format($payment->installment->amount, 0) }}</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <strong>Warning:</strong> Changing the payment amount or status will affect the installment's paid amount and status.
                    </div>

                    <div class="mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i> Update Payment
                        </button>
                        <a href="{{ route('payments.show', $payment) }}" class="btn btn-info">
                            <i class="fas fa-eye me-2"></i> View Details
                        </a>
                        <a href="{{ route('payments.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times me-2"></i> Cancel
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
    $(document).ready(function() {
        // Calculate new total paid when amount changes
        $('#amount').on('input', function() {
            const originalAmount = parseFloat('{{ $payment->amount }}');
            const newAmount = parseFloat($(this).val()) || 0;
            const previouslyPaid = parseFloat('{{ $payment->installment->paid_amount - $payment->amount }}');
            const newTotalPaid = previouslyPaid + newAmount;

            // Update new total paid display
            $('#newTotalPaid').text('Rs. ' + newTotalPaid.toFixed(2));

            // Update progress bar
            const installmentAmount = parseFloat('{{ $payment->installment->amount }}');
            const progressPercent = (newTotalPaid / installmentAmount) * 100;
            $('#progressBar').css('width', progressPercent + '%');
            $('#progressPercent').text(Math.round(progressPercent) + '%');

            // Validate amount doesn't exceed installment amount
            if (newTotalPaid > installmentAmount) {
                alert('Total paid amount cannot exceed installment amount: Rs. ' + installmentAmount.toFixed(2));
                $(this).val((installmentAmount - previouslyPaid).toFixed(2));
            }
        });

        // Validate reference number based on payment method
        $('#payment_method').on('change', function() {
            const method = $(this).val();
            const referenceField = $('#reference_number');

            if (method === 'cash') {
                referenceField.removeAttr('required');
            } else {
                referenceField.attr('required', 'required');
            }
        });

        // Form validation
        $('#paymentForm').validate({
            rules: {
                amount: {
                    required: true,
                    min: 1,
                    max: function() {
                        const installmentAmount = parseFloat('{{ $payment->installment->amount }}');
                        const previouslyPaid = parseFloat('{{ $payment->installment->paid_amount - $payment->amount }}');
                        return installmentAmount - previouslyPaid;
                    }
                },
                payment_date: 'required',
                payment_method: 'required',
                reference_number: {
                    required: function() {
                        const method = $('#payment_method').val();
                        return method !== 'cash';
                    }
                },
                status: 'required'
            },
            messages: {
                amount: {
                    required: 'Please enter payment amount',
                    min: 'Amount must be at least Rs. 1',
                    max: 'Amount cannot exceed remaining installment amount'
                },
                payment_date: 'Please select payment date',
                payment_method: 'Please select payment method',
                reference_number: 'Reference number is required for this payment method',
                status: 'Please select payment status'
            },
            errorElement: 'div',
            errorClass: 'invalid-feedback',
            highlight: function(element) {
                $(element).addClass('is-invalid');
            },
            unhighlight: function(element) {
                $(element).removeClass('is-invalid');
            }
        });
    });
</script>
@endpush
@endsection
