@extends('layouts.app')

@section('title', 'Payments')
@section('page-title', 'Payment Management')

@section('content')
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">All Payments</h5>
        <div>
            <a href="{{ route('payments.create') }}" class="btn btn-primary">
                <i class="fas fa-plus me-2"></i> Record New Payment
            </a>
            <button class="btn btn-success" onclick="exportToExcel()">
                <i class="fas fa-download me-2"></i> Export
            </button>
        </div>
    </div>

    <div class="card-body">
        <!-- Filters -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="input-group">
                    <span class="input-group-text"><i class="fas fa-search"></i></span>
                    <input type="text" id="searchInput" class="form-control" placeholder="Search payments...">
                </div>
            </div>
            <div class="col-md-2">
                <select id="statusFilter" class="form-control">
                    <option value="">All Status</option>
                    <option value="completed">Completed</option>
                    <option value="pending">Pending</option>
                    <option value="failed">Failed</option>
                    <option value="refunded">Refunded</option>
                </select>
            </div>
            <div class="col-md-2">
                <select id="methodFilter" class="form-control">
                    <option value="">All Methods</option>
                    <option value="cash">Cash</option>
                    <option value="bank_transfer">Bank Transfer</option>
                    <option value="cheque">Cheque</option>
                    <option value="online">Online</option>
                </select>
            </div>
            <div class="col-md-2">
                <input type="date" id="startDateFilter" class="form-control" placeholder="From Date">
            </div>
            <div class="col-md-2">
                <input type="date" id="endDateFilter" class="form-control" placeholder="To Date">
            </div>
            <div class="col-md-1">
                <button class="btn btn-outline-secondary w-100" onclick="resetFilters()">
                    <i class="fas fa-redo"></i>
                </button>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card bg-primary text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-0">Total Payments</h6>
                                <h3 class="mb-0">{{ $payments->total() }}</h3>
                            </div>
                            <i class="fas fa-credit-card fa-2x opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-success text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-0">Total Amount</h6>
                                <h3 class="mb-0">Rs. {{ number_format($payments->sum('amount'), 0) }}</h3>
                            </div>
                            <i class="fas fa-money-bill-wave fa-2x opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-info text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-0">Completed</h6>
                                <h3 class="mb-0">{{ $payments->where('status', 'completed')->count() }}</h3>
                            </div>
                            <i class="fas fa-check-circle fa-2x opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-warning text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-0">Pending</h6>
                                <h3 class="mb-0">{{ $payments->where('status', 'pending')->count() }}</h3>
                            </div>
                            <i class="fas fa-clock fa-2x opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Payments Table -->
        <div class="table-responsive">
            <table class="table table-hover" id="paymentsTable">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Transaction ID</th>
                        <th>Customer</th>
                        <th>Property</th>
                        <th>Installment</th>
                        <th>Date</th>
                        <th>Amount</th>
                        <th>Method</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($payments as $payment)
                    <tr class="payment-row"
                        data-status="{{ $payment->status }}"
                        data-method="{{ $payment->payment_method }}"
                        data-date="{{ $payment->payment_date->format('Y-m-d') }}">
                        <td>{{ $loop->iteration }}</td>
                        <td>
                            <code class="text-primary">{{ $payment->transaction_id }}</code>
                            @if($payment->reference_number)
                                <div class="small text-muted">Ref: {{ $payment->reference_number }}</div>
                            @endif
                        </td>
                        <td>
                            <div class="d-flex align-items-center">
                                <div class="avatar bg-info text-white rounded-circle d-flex align-items-center justify-content-center me-2"
                                     style="width: 30px; height: 30px; font-size: 12px;">
                                    {{ substr($payment->customer->name, 0, 1) }}
                                </div>
                                <div>
                                    <strong>{{ $payment->customer->name }}</strong>
                                    <div class="small text-muted">{{ $payment->customer->phone }}</div>
                                </div>
                            </div>
                        </td>
                        <td>
                            <strong>{{ $payment->installment->property->name }}</strong>
                            <div class="small text-muted">
                                {{ Str::limit($payment->installment->property->location, 20) }}
                            </div>
                        </td>
                        <td>
                            <span class="badge bg-secondary">
                                #{{ $payment->installment->installment_number }}
                            </span>
                            <div class="small text-muted">
                                Due: {{ $payment->installment->due_date->format('d/m/Y') }}
                            </div>
                        </td>
                        <td>
                            {{ $payment->payment_date->format('d M Y') }}
                            <div class="small text-muted">
                                {{ $payment->created_at->format('h:i A') }}
                            </div>
                        </td>
                        <td>
                            <span class="fw-bold text-success">
                                Rs. {{ number_format($payment->amount, 2) }}
                            </span>
                        </td>
                        <td>
                            @php
                                $methodColors = [
                                    'cash' => 'secondary',
                                    'bank_transfer' => 'primary',
                                    'cheque' => 'info',
                                    'online' => 'success'
                                ];
                                $methodIcons = [
                                    'cash' => 'money-bill',
                                    'bank_transfer' => 'university',
                                    'cheque' => 'file-invoice',
                                    'online' => 'globe'
                                ];
                            @endphp
                            <span class="badge bg-{{ $methodColors[$payment->payment_method] ?? 'secondary' }}">
                                <i class="fas fa-{{ $methodIcons[$payment->payment_method] ?? 'money-bill' }} me-1"></i>
                                {{ ucfirst(str_replace('_', ' ', $payment->payment_method)) }}
                            </span>
                        </td>
                        <td>
                            @php
                                $statusColors = [
                                    'completed' => 'success',
                                    'pending' => 'warning',
                                    'failed' => 'danger',
                                    'refunded' => 'info'
                                ];
                                $statusIcons = [
                                    'completed' => 'check-circle',
                                    'pending' => 'clock',
                                    'failed' => 'times-circle',
                                    'refunded' => 'exchange-alt'
                                ];
                            @endphp
                            <span class="badge bg-{{ $statusColors[$payment->status] ?? 'secondary' }}">
                                <i class="fas fa-{{ $statusIcons[$payment->status] ?? 'circle' }} me-1"></i>
                                {{ ucfirst($payment->status) }}
                            </span>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm" role="group">
                                <a href="{{ route('payments.show', $payment) }}"
                                   class="btn btn-info" title="View Details">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <a href="{{ route('payments.edit', $payment) }}"
                                   class="btn btn-warning" title="Edit Payment">
                                    <i class="fas fa-edit"></i>
                                </a>
                                @if($payment->status != 'refunded')
                                    <button type="button" class="btn btn-danger"
                                            onclick="confirmRefund({{ $payment->id }})"
                                            title="Mark as Refunded">
                                        <i class="fas fa-undo"></i>
                                    </button>
                                @endif
                                <form action="{{ route('payments.destroy', $payment) }}" method="POST"
                                      class="d-inline delete-form">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" class="btn btn-dark"
                                            onclick="return confirm('Delete this payment record?')"
                                            title="Delete">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </form>
                            </div>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="10" class="text-center py-5">
                            <div class="empty-state">
                                <i class="fas fa-credit-card fa-3x text-muted mb-3"></i>
                                <h5>No Payments Found</h5>
                                <p class="text-muted">No payment records have been created yet.</p>
                                <a href="{{ route('payments.create') }}" class="btn btn-primary">
                                    <i class="fas fa-plus me-2"></i> Record First Payment
                                </a>
                            </div>
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        @if($payments->hasPages())
        <div class="d-flex justify-content-between align-items-center mt-3">
            <div>
                Showing {{ $payments->firstItem() }} to {{ $payments->lastItem() }} of {{ $payments->total() }} entries
            </div>
            <div>
                {{ $payments->links() }}
            </div>
        </div>
        @endif
    </div>
</div>

<!-- Refund Modal -->
<div class="modal fade" id="refundModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Mark Payment as Refunded</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="refundForm" method="POST">
                    @csrf
                    @method('PUT')
                    <input type="hidden" name="status" value="refunded">
                    <input type="hidden" name="payment_id" id="refundPaymentId">

                    <div class="mb-3">
                        <label for="refund_date" class="form-label">Refund Date</label>
                        <input type="date" class="form-control" id="refund_date" name="refund_date"
                               value="{{ date('Y-m-d') }}" required>
                    </div>

                    <div class="mb-3">
                        <label for="refund_reason" class="form-label">Refund Reason</label>
                        <textarea class="form-control" id="refund_reason" name="remarks"
                                  rows="3" placeholder="Enter reason for refund" required></textarea>
                    </div>

                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        This action cannot be undone. The payment amount will be deducted from the installment.
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="submit" form="refundForm" class="btn btn-danger">Confirm Refund</button>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
    // Search functionality
    $('#searchInput').on('keyup', function() {
        const value = $(this).val().toLowerCase();
        $('#paymentsTable tbody tr').filter(function() {
            $(this).toggle($(this).text().toLowerCase().indexOf(value) > -1);
        });
    });

    // Filter by status
    $('#statusFilter').on('change', function() {
        filterPayments();
    });

    // Filter by method
    $('#methodFilter').on('change', function() {
        filterPayments();
    });

    // Filter by date range
    $('#startDateFilter, #endDateFilter').on('change', function() {
        filterPayments();
    });

    function filterPayments() {
        const status = $('#statusFilter').val();
        const method = $('#methodFilter').val();
        const startDate = $('#startDateFilter').val();
        const endDate = $('#endDateFilter').val();

        $('#paymentsTable tbody tr').each(function() {
            const showStatus = !status || $(this).data('status') === status;
            const showMethod = !method || $(this).data('method') === method;
            const paymentDate = $(this).data('date');
            const showDate = (!startDate || paymentDate >= startDate) &&
                            (!endDate || paymentDate <= endDate);

            $(this).toggle(showStatus && showMethod && showDate);
        });
    }

    // Reset filters
    function resetFilters() {
        $('#searchInput').val('');
        $('#statusFilter').val('');
        $('#methodFilter').val('');
        $('#startDateFilter').val('');
        $('#endDateFilter').val('');
        $('#paymentsTable tbody tr').show();
    }

    // Confirm refund
    function confirmRefund(paymentId) {
        $('#refundPaymentId').val(paymentId);
        $('#refundForm').attr('action', `/payments/${paymentId}`);
        new bootstrap.Modal(document.getElementById('refundModal')).show();
    }

    // Export to Excel
    function exportToExcel() {
        const rows = [];
        const headers = ['Transaction ID', 'Customer', 'Property', 'Installment #', 'Payment Date', 'Amount', 'Method', 'Status', 'Reference'];

        $('#paymentsTable tbody tr').each(function() {
            const row = [];
            row.push($(this).find('td:nth-child(2) code').text().trim());
            row.push($(this).find('td:nth-child(3) strong').text().trim());
            row.push($(this).find('td:nth-child(4) strong').text().trim());
            row.push($(this).find('td:nth-child(5) span').text().replace('#', '').trim());
            row.push($(this).find('td:nth-child(6)').text().split('\n')[0].trim());
            row.push($(this).find('td:nth-child(7) span').text().replace('Rs.', '').trim());
            row.push($(this).find('td:nth-child(8) span').text().trim());
            row.push($(this).find('td:nth-child(9) span').text().trim());
            row.push($(this).find('td:nth-child(2) .small').text().replace('Ref:', '').trim() || 'N/A');

            rows.push(row.join(','));
        });

        const csvContent = [headers.join(','), ...rows].join('\n');
        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `payments_${new Date().toISOString().split('T')[0]}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
    }

    // Initialize DataTable
    $(document).ready(function() {
        $('#paymentsTable').DataTable({
            paging: false,
            searching: false,
            info: false,
            order: [[5, 'desc']]
        });
    });
</script>
@endpush

@push('styles')
<style>
    .avatar {
        font-weight: bold;
    }

    .badge {
        min-width: 100px;
        text-align: center;
    }

    .payment-row:hover {
        background-color: rgba(67, 97, 238, 0.05);
        cursor: pointer;
    }

    .empty-state {
        opacity: 0.7;
    }

    code {
        font-size: 12px;
        background: #f8f9fa;
        padding: 2px 6px;
        border-radius: 4px;
        border: 1px solid #e9ecef;
    }

    .card.bg-primary,
    .card.bg-success,
    .card.bg-info,
    .card.bg-warning {
        transition: transform 0.3s;
    }

    .card.bg-primary:hover,
    .card.bg-success:hover,
    .card.bg-info:hover,
    .card.bg-warning:hover {
        transform: translateY(-5px);
    }
</style>
@endpush
@endsection
