@extends('layouts.app')

@section('title', $property->name)
@section('page-title', $property->name)

@section('content')
<div class="row">
    <!-- Property Details -->
    <div class="col-md-4">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Property Details</h5>
                <div class="dropdown">
                    <button class="btn btn-sm btn-outline-secondary dropdown-toggle" type="button"
                            data-bs-toggle="dropdown">
                        <i class="fas fa-cog"></i>
                    </button>
                    <ul class="dropdown-menu">
                        <li>
                            <a class="dropdown-item" href="{{ route('properties.edit', $property) }}">
                                <i class="fas fa-edit me-2"></i> Edit
                            </a>
                        </li>
                        <li>
                            <a class="dropdown-item" href="{{ route('payments.create', ['installment_id' => 'first']) }}">
                                <i class="fas fa-credit-card me-2"></i> Record Payment
                            </a>
                        </li>
                        <li><hr class="dropdown-divider"></li>
                        <li>
                            <form action="{{ route('properties.destroy', $property) }}" method="POST"
                                  class="dropdown-item p-0">
                                @csrf
                                @method('DELETE')
                                <button type="submit" class="btn btn-link text-danger p-0"
                                        onclick="return confirm('Are you sure? This will delete all related installments and payments.')">
                                    <i class="fas fa-trash me-2"></i> Delete
                                </button>
                            </form>
                        </li>
                    </ul>
                </div>
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <small class="text-muted">Property Name</small>
                    <p class="mb-2"><strong>{{ $property->name }}</strong></p>
                </div>

                <div class="mb-3">
                    <small class="text-muted">Location</small>
                    <p class="mb-2">{{ $property->location }}</p>
                </div>

                <div class="mb-3">
                    <small class="text-muted">Description</small>
                    <p class="mb-2">{{ $property->description ?: 'No description provided' }}</p>
                </div>

                <hr>

                <div class="row">
                    <div class="col-6">
                        <small class="text-muted">Total Price</small>
                        <p class="mb-1"><strong>Rs. {{ number_format($property->total_price, 2) }}</strong></p>
                    </div>
                    <div class="col-6">
                        <small class="text-muted">Down Payment</small>
                        <p class="mb-1"><strong>Rs. {{ number_format($property->down_payment, 2) }}</strong></p>
                    </div>
                </div>

                <div class="row mt-2">
                    <div class="col-6">
                        <small class="text-muted">Installment Amount</small>
                        <p class="mb-1"><strong>Rs. {{ number_format($property->installment_amount, 2) }}</strong></p>
                    </div>
                    <div class="col-6">
                        <small class="text-muted">Total Installments</small>
                        <p class="mb-1"><strong>{{ $property->total_installments }}</strong></p>
                    </div>
                </div>

                <div class="row mt-2">
                    <div class="col-6">
                        <small class="text-muted">Installment Years</small>
                        <p class="mb-1"><strong>{{ $property->installment_years }} Years</strong></p>
                    </div>
                    <div class="col-6">
                        <small class="text-muted">Status</small>
                        <p class="mb-1">
                            <span class="badge bg-{{ $property->status == 'active' ? 'success' : ($property->status == 'completed' ? 'primary' : 'danger') }}">
                                {{ ucfirst($property->status) }}
                            </span>
                        </p>
                    </div>
                </div>

                <div class="row mt-2">
                    <div class="col-12">
                        <small class="text-muted">Remaining Amount</small>
                        <p class="mb-1"><strong>Rs. {{ number_format($property->remaining_amount, 2) }}</strong></p>
                    </div>
                </div>

                <!-- Progress Bar -->
                <div class="mt-4">
                    <small class="text-muted">Payment Progress</small>
                    <div class="progress" style="height: 20px;">
                        <div class="progress-bar bg-success" role="progressbar"
                             style="width: {{ $property->progress }}%;"
                             aria-valuenow="{{ $property->progress }}"
                             aria-valuemin="0" aria-valuemax="100">
                            {{ round($property->progress) }}%
                        </div>
                    </div>
                    <div class="text-center mt-1">
                        <small>
                            {{ $property->installments->where('status', 'paid')->count() }} /
                            {{ $property->total_installments }} installments paid
                        </small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Associated Customers -->
        <div class="card mt-4">
            <div class="card-header">
                <h6 class="mb-0">Associated Customers</h6>
            </div>
            <div class="card-body">
                @if($property->customers->count() > 0)
                    @foreach($property->customers as $customer)
                        <div class="d-flex align-items-center mb-3">
                            <div class="flex-shrink-0">
                                <div class="avatar bg-primary text-white rounded-circle d-flex align-items-center justify-content-center"
                                     style="width: 40px; height: 40px;">
                                    {{ substr($customer->name, 0, 1) }}
                                </div>
                            </div>
                            <div class="flex-grow-1 ms-3">
                                <h6 class="mb-0">{{ $customer->name }}</h6>
                                <small class="text-muted">{{ $customer->phone }}</small>
                            </div>
                            <div class="flex-shrink-0">
                                <a href="{{ route('customers.show', $customer) }}" class="btn btn-sm btn-outline-primary">
                                    <i class="fas fa-eye"></i>
                                </a>
                            </div>
                        </div>
                    @endforeach
                @else
                    <p class="text-muted mb-0">No customers associated with this property.</p>
                @endif
            </div>
        </div>
    </div>

    <!-- Installments Table -->
    <div class="col-md-8">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Installment Schedule</h5>
                <div>
                    <div class="btn-group" role="group">
                        <button type="button" class="btn btn-sm btn-outline-secondary"
                                onclick="filterInstallments('all')">All</button>
                        <button type="button" class="btn btn-sm btn-outline-success"
                                onclick="filterInstallments('paid')">Paid</button>
                        <button type="button" class="btn btn-sm btn-outline-warning"
                                onclick="filterInstallments('pending')">Pending</button>
                        <button type="button" class="btn btn-sm btn-outline-danger"
                                onclick="filterInstallments('overdue')">Overdue</button>
                    </div>
                </div>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover" id="installmentsTable">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Due Date</th>
                                <th>Amount</th>
                                <th>Paid</th>
                                <th>Remaining</th>
                                <th>Status</th>
                                <th>Customer</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($installments as $installment)
                            <tr class="installment-row" data-status="{{ $installment->status }}">
                                <td>{{ $installment->installment_number }}</td>
                                <td>
                                    {{ $installment->due_date->format('d M Y') }}
                                    @if($installment->isOverdue())
                                        <br><small class="text-danger">Overdue</small>
                                    @endif
                                </td>
                                <td>Rs. {{ number_format($installment->amount, 2) }}</td>
                                <td>Rs. {{ number_format($installment->paid_amount, 2) }}</td>
                                <td>Rs. {{ number_format($installment->remaining_amount, 2) }}</td>
                                <td>
                                    @php
                                        $statusColors = [
                                            'paid' => 'success',
                                            'pending' => 'warning',
                                            'overdue' => 'danger',
                                            'partial' => 'info'
                                        ];
                                    @endphp
                                    <span class="badge bg-{{ $statusColors[$installment->status] ?? 'secondary' }} status-badge">
                                        {{ ucfirst($installment->status) }}
                                    </span>
                                </td>
                                <td>
                                    <a href="{{ route('customers.show', $installment->customer) }}"
                                       class="text-decoration-none">
                                        {{ $installment->customer->name }}
                                    </a>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm" role="group">
                                        <a href="{{ route('payments.create', ['installment_id' => $installment->id]) }}"
                                           class="btn btn-outline-primary" title="Record Payment">
                                            <i class="fas fa-credit-card"></i>
                                        </a>
                                        <a href="{{ route('installments.show', $installment) }}"
                                           class="btn btn-outline-info" title="View Details">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <a href="{{ route('installments.edit', $installment) }}"
                                           class="btn btn-outline-warning" title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>

                <div class="d-flex justify-content-center mt-3">
                    {{ $installments->links() }}
                </div>
            </div>
        </div>

        <!-- Payment Statistics -->
        <div class="row mt-4">
            <div class="col-md-4">
                <div class="card bg-success text-white">
                    <div class="card-body text-center">
                        <h1 class="mb-0">
                            {{ $property->installments->where('status', 'paid')->count() }}
                        </h1>
                        <p class="mb-0">Paid Installments</p>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card bg-warning text-white">
                    <div class="card-body text-center">
                        <h1 class="mb-0">
                            {{ $property->installments->whereIn('status', ['pending', 'partial'])->count() }}
                        </h1>
                        <p class="mb-0">Pending Installments</p>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card bg-danger text-white">
                    <div class="card-body text-center">
                        <h1 class="mb-0">
                            {{ $property->installments->where('status', 'overdue')->count() }}
                        </h1>
                        <p class="mb-0">Overdue Installments</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recent Payments -->
        @if($property->installments->flatMap->payments->count() > 0)
        <div class="card mt-4">
            <div class="card-header">
                <h6 class="mb-0">Recent Payments</h6>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Amount</th>
                                <th>Method</th>
                                <th>Installment #</th>
                                <th>Reference</th>
                            </tr>
                        </thead>
                        <tbody>
                            @php
                                $recentPayments = $property->installments
                                    ->flatMap->payments
                                    ->sortByDesc('payment_date')
                                    ->take(5);
                            @endphp
                            @foreach($recentPayments as $payment)
                            <tr>
                                <td>{{ $payment->payment_date->format('d/m/Y') }}</td>
                                <td>Rs. {{ number_format($payment->amount, 2) }}</td>
                                <td>
                                    <span class="badge bg-info">
                                        {{ ucfirst(str_replace('_', ' ', $payment->payment_method)) }}
                                    </span>
                                </td>
                                <td>#{{ $payment->installment->installment_number }}</td>
                                <td>{{ $payment->reference_number ?: 'N/A' }}</td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                @if($recentPayments->count() < $property->installments->flatMap->payments->count())
                    <div class="text-center">
                        <a href="{{ route('payments.index', ['property_id' => $property->id]) }}"
                           class="btn btn-sm btn-outline-primary">
                            View All Payments
                        </a>
                    </div>
                @endif
            </div>
        </div>
        @endif
    </div>
</div>

@push('scripts')
<script>
    // Filter installments by status
    function filterInstallments(status) {
        const rows = document.querySelectorAll('.installment-row');

        rows.forEach(row => {
            if (status === 'all' || row.dataset.status === status) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        });

        // Update button active state
        document.querySelectorAll('.btn-group .btn').forEach(btn => {
            btn.classList.remove('active');
        });
        event.target.classList.add('active');
    }

    // Initialize with all installments shown
    $(document).ready(function() {
        // Add export buttons
        const table = $('#installmentsTable').DataTable({
            dom: 'Bfrtip',
            buttons: [
                'copy', 'csv', 'excel', 'pdf', 'print'
            ],
            pageLength: 10,
            order: [[1, 'asc']]
        });
    });
</script>
@endpush

@push('styles')
<style>
    .avatar {
        font-weight: bold;
        font-size: 16px;
    }

    .status-badge {
        min-width: 70px;
        display: inline-block;
        text-align: center;
    }

    .installment-row:hover {
        background-color: rgba(67, 97, 238, 0.05);
        cursor: pointer;
    }
</style>
@endpush
@endsection
