@extends('layouts.app')

@section('title', 'Installments Report')
@section('page-title', 'Installments Report')

@section('content')
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">Installments Report</h5>
        <div>
            <button class="btn btn-success" onclick="exportToExcel()">
                <i class="fas fa-download me-2"></i> Export
            </button>
            <button class="btn btn-primary" onclick="window.print()">
                <i class="fas fa-print me-2"></i> Print
            </button>
        </div>
    </div>

    <div class="card-body">
        <!-- Filters -->
        <form method="GET" action="{{ route('reports.installments') }}" id="filterForm">
            <div class="row mb-4">
                <div class="col-md-2">
                    <label for="status" class="form-label">Status</label>
                    <select class="form-control" id="status" name="status">
                        <option value="all" {{ request('status') == 'all' ? 'selected' : '' }}>All Status</option>
                        <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>Pending</option>
                        <option value="paid" {{ request('status') == 'paid' ? 'selected' : '' }}>Paid</option>
                        <option value="overdue" {{ request('status') == 'overdue' ? 'selected' : '' }}>Overdue</option>
                        <option value="partial" {{ request('status') == 'partial' ? 'selected' : '' }}>Partial</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <label for="property_id" class="form-label">Property</label>
                    <select class="form-control" id="property_id" name="property_id">
                        <option value="">All Properties</option>
                        @foreach($properties as $property)
                            <option value="{{ $property->id }}" {{ request('property_id') == $property->id ? 'selected' : '' }}>
                                {{ $property->name }}
                            </option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2">
                    <label for="start_date" class="form-label">Start Date</label>
                    <input type="date" class="form-control" id="start_date" name="start_date"
                           value="{{ request('start_date') }}">
                </div>
                <div class="col-md-2">
                    <label for="end_date" class="form-label">End Date</label>
                    <input type="date" class="form-control" id="end_date" name="end_date"
                           value="{{ request('end_date') }}">
                </div>
                <div class="col-md-2">
                    <label class="form-label">&nbsp;</label>
                    <div class="d-flex">
                        <button type="submit" class="btn btn-primary me-2">
                            <i class="fas fa-filter"></i> Filter
                        </button>
                        <a href="{{ route('reports.installments') }}" class="btn btn-secondary">
                            <i class="fas fa-redo"></i>
                        </a>
                    </div>
                </div>
            </div>
        </form>

        <!-- Summary Statistics -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card bg-primary text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">{{ $installments->count() }}</h3>
                        <small>Total Installments</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-success text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">Rs. {{ number_format($totalAmount, 0) }}</h3>
                        <small>Total Amount</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-info text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">Rs. {{ number_format($totalPaid, 0) }}</h3>
                        <small>Total Paid</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-warning text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">{{ $overdueCount }}</h3>
                        <small>Overdue Installments</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Installments Table -->
        <div class="table-responsive">
            <table class="table table-hover" id="installmentsTable">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Property</th>
                        <th>Customer</th>
                        <th>Installment #</th>
                        <th>Due Date</th>
                        <th>Amount</th>
                        <th>Paid</th>
                        <th>Remaining</th>
                        <th>Progress</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($installments as $installment)
                    <tr>
                        <td>{{ $loop->iteration }}</td>
                        <td>
                            <strong>{{ $installment->property->name }}</strong>
                            <div class="small text-muted">{{ Str::limit($installment->property->location, 30) }}</div>
                        </td>
                        <td>
                            <div class="d-flex align-items-center">
                                <div class="avatar bg-info text-white rounded-circle d-flex align-items-center justify-content-center me-2"
                                     style="width: 30px; height: 30px; font-size: 12px;">
                                    {{ substr($installment->customer->name, 0, 1) }}
                                </div>
                                <div>
                                    <strong>{{ $installment->customer->name }}</strong>
                                    <div class="small text-muted">{{ $installment->customer->phone }}</div>
                                </div>
                            </div>
                        </td>
                        <td>
                            <span class="badge bg-secondary">#{{ $installment->installment_number }}</span>
                        </td>
                        <td>
                            {{ $installment->due_date->format('d M Y') }}
                            @if($installment->isOverdue())
                                <br><small class="text-danger">Overdue</small>
                            @endif
                        </td>
                        <td>
                            <strong>Rs. {{ number_format($installment->amount, 2) }}</strong>
                        </td>
                        <td>
                            <span class="text-success">
                                Rs. {{ number_format($installment->paid_amount, 2) }}
                            </span>
                        </td>
                        <td>
                            <span class="text-warning">
                                Rs. {{ number_format($installment->remaining_amount, 2) }}
                            </span>
                        </td>
                        <td>
                            @php
                                $progress = ($installment->paid_amount / $installment->amount) * 100;
                            @endphp
                            <div class="progress" style="height: 10px;">
                                <div class="progress-bar bg-success" style="width: {{ $progress }}%"></div>
                            </div>
                            <small>{{ round($progress) }}%</small>
                        </td>
                        <td>
                            @php
                                $statusColors = [
                                    'paid' => 'success',
                                    'pending' => 'warning',
                                    'overdue' => 'danger',
                                    'partial' => 'info'
                                ];
                            @endphp
                            <span class="badge bg-{{ $statusColors[$installment->status] ?? 'secondary' }}">
                                {{ ucfirst($installment->status) }}
                            </span>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>

        @if($installments->isEmpty())
        <div class="text-center py-5">
            <i class="fas fa-calendar-alt fa-3x text-muted mb-3"></i>
            <h5>No Installments Found</h5>
            <p class="text-muted">No installments match your search criteria.</p>
        </div>
        @endif
    </div>
</div>

@push('scripts')
<script>
    function exportToExcel() {
        const rows = [];
        const headers = ['Property', 'Customer', 'Customer Phone', 'Installment #', 'Due Date', 'Amount', 'Paid Amount', 'Remaining Amount', 'Progress %', 'Status', 'Overdue'];

        $('#installmentsTable tbody tr').each(function() {
            const row = [];
            row.push($(this).find('td:nth-child(2) strong').text().trim());
            row.push($(this).find('td:nth-child(3) strong').text().trim());
            row.push($(this).find('td:nth-child(3) .small').text().trim());
            row.push($(this).find('td:nth-child(4) span').text().replace('#', '').trim());
            row.push($(this).find('td:nth-child(5)').text().split('\n')[0].trim());
            row.push($(this).find('td:nth-child(6) strong').text().replace('Rs.', '').trim());
            row.push($(this).find('td:nth-child(7) span').text().replace('Rs.', '').trim());
            row.push($(this).find('td:nth-child(8) span').text().replace('Rs.', '').trim());
            row.push($(this).find('td:nth-child(9) small').text().replace('%', '').trim());
            row.push($(this).find('td:nth-child(10) span').text().trim());
            row.push($(this).find('td:nth-child(5) .small').text().trim() ? 'Yes' : 'No');

            rows.push(row.join(','));
        });

        const csvContent = [headers.join(','), ...rows].join('\n');
        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `installments_report_${new Date().toISOString().split('T')[0]}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
    }
</script>
@endpush
@endsection
