@extends('layouts.app')

@section('title', 'Payments Report')
@section('page-title', 'Payments Report')

@section('content')
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">Payments Report</h5>
        <div>
            <button class="btn btn-success" onclick="exportToExcel()">
                <i class="fas fa-download me-2"></i> Export
            </button>
            <button class="btn btn-primary" onclick="window.print()">
                <i class="fas fa-print me-2"></i> Print
            </button>
        </div>
    </div>

    <div class="card-body">
        <!-- Filters -->
        <form method="GET" action="{{ route('reports.payments') }}" id="filterForm">
            <div class="row mb-4">
                <div class="col-md-3">
                    <label for="start_date" class="form-label">Start Date</label>
                    <input type="date" class="form-control" id="start_date" name="start_date"
                           value="{{ request('start_date') }}">
                </div>
                <div class="col-md-3">
                    <label for="end_date" class="form-label">End Date</label>
                    <input type="date" class="form-control" id="end_date" name="end_date"
                           value="{{ request('end_date') }}">
                </div>
                <div class="col-md-2">
                    <label for="status" class="form-label">Status</label>
                    <select class="form-control" id="status" name="status">
                        <option value="all" {{ request('status') == 'all' ? 'selected' : '' }}>All Status</option>
                        <option value="completed" {{ request('status') == 'completed' ? 'selected' : '' }}>Completed</option>
                        <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>Pending</option>
                        <option value="failed" {{ request('status') == 'failed' ? 'selected' : '' }}>Failed</option>
                        <option value="refunded" {{ request('status') == 'refunded' ? 'selected' : '' }}>Refunded</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <label for="method" class="form-label">Method</label>
                    <select class="form-control" id="method" name="method">
                        <option value="all" {{ request('method') == 'all' ? 'selected' : '' }}>All Methods</option>
                        <option value="cash" {{ request('method') == 'cash' ? 'selected' : '' }}>Cash</option>
                        <option value="bank_transfer" {{ request('method') == 'bank_transfer' ? 'selected' : '' }}>Bank Transfer</option>
                        <option value="cheque" {{ request('method') == 'cheque' ? 'selected' : '' }}>Cheque</option>
                        <option value="online" {{ request('method') == 'online' ? 'selected' : '' }}>Online</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">&nbsp;</label>
                    <div class="d-flex">
                        <button type="submit" class="btn btn-primary me-2">
                            <i class="fas fa-filter"></i> Filter
                        </button>
                        <a href="{{ route('reports.payments') }}" class="btn btn-secondary">
                            <i class="fas fa-redo"></i>
                        </a>
                    </div>
                </div>
            </div>
        </form>

        <!-- Summary Statistics -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card bg-primary text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">{{ $totalCount }}</h3>
                        <small>Total Payments</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-success text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">{{ $completedCount }}</h3>
                        <small>Completed</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-info text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">Rs. {{ number_format($totalAmount, 0) }}</h3>
                        <small>Total Amount</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-warning text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">{{ $payments->where('status', 'pending')->count() }}</h3>
                        <small>Pending Payments</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Payments Table -->
        <div class="table-responsive">
            <table class="table table-hover" id="paymentsTable">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Transaction ID</th>
                        <th>Customer</th>
                        <th>Property</th>
                        <th>Date</th>
                        <th>Amount</th>
                        <th>Method</th>
                        <th>Reference</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($payments as $payment)
                    <tr>
                        <td>{{ $loop->iteration }}</td>
                        <td>
                            <code>{{ $payment->transaction_id }}</code>
                            @if($payment->remarks)
                            <div class="small text-muted">{{ Str::limit($payment->remarks, 30) }}</div>
                            @endif
                        </td>
                        <td>
                            <div class="d-flex align-items-center">
                                <div class="avatar bg-info text-white rounded-circle d-flex align-items-center justify-content-center me-2"
                                     style="width: 30px; height: 30px; font-size: 12px;">
                                    {{ substr($payment->customer->name, 0, 1) }}
                                </div>
                                <div>
                                    <strong>{{ $payment->customer->name }}</strong>
                                    <div class="small text-muted">{{ $payment->customer->phone }}</div>
                                </div>
                            </div>
                        </td>
                        <td>
                            <strong>{{ $payment->installment->property->name }}</strong>
                            <div class="small text-muted">
                                Installment #{{ $payment->installment->installment_number }}
                            </div>
                        </td>
                        <td>{{ $payment->payment_date->format('d M Y') }}</td>
                        <td>
                            <strong class="text-success">
                                Rs. {{ number_format($payment->amount, 2) }}
                            </strong>
                        </td>
                        <td>
                            <span class="badge bg-primary">
                                {{ ucfirst(str_replace('_', ' ', $payment->payment_method)) }}
                            </span>
                        </td>
                        <td>
                            @if($payment->reference_number)
                                <code>{{ $payment->reference_number }}</code>
                            @else
                                <span class="text-muted">N/A</span>
                            @endif
                        </td>
                        <td>
                            @php
                                $statusColors = [
                                    'completed' => 'success',
                                    'pending' => 'warning',
                                    'failed' => 'danger',
                                    'refunded' => 'info'
                                ];
                            @endphp
                            <span class="badge bg-{{ $statusColors[$payment->status] ?? 'secondary' }}">
                                {{ ucfirst($payment->status) }}
                            </span>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>

        @if($payments->isEmpty())
        <div class="text-center py-5">
            <i class="fas fa-credit-card fa-3x text-muted mb-3"></i>
            <h5>No Payments Found</h5>
            <p class="text-muted">No payments match your search criteria.</p>
        </div>
        @endif
    </div>
</div>

@push('scripts')
<script>
    function exportToExcel() {
        const rows = [];
        const headers = ['Transaction ID', 'Customer', 'Property', 'Installment #', 'Payment Date', 'Amount', 'Payment Method', 'Reference Number', 'Status', 'Remarks'];

        $('#paymentsTable tbody tr').each(function() {
            const row = [];
            row.push($(this).find('td:nth-child(2) code').text().trim());
            row.push($(this).find('td:nth-child(3) strong').text().trim());
            row.push($(this).find('td:nth-child(4) strong').text().trim());
            row.push($(this).find('td:nth-child(4) .small').text().replace('Installment #', '').trim());
            row.push($(this).find('td:nth-child(5)').text().trim());
            row.push($(this).find('td:nth-child(6) strong').text().replace('Rs.', '').trim());
            row.push($(this).find('td:nth-child(7) span').text().trim());
            row.push($(this).find('td:nth-child(8)').text().trim());
            row.push($(this).find('td:nth-child(9) span').text().trim());
            row.push($(this).find('td:nth-child(2) .small').text().trim() || 'N/A');

            rows.push(row.join(','));
        });

        const csvContent = [headers.join(','), ...rows].join('\n');
        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `payments_report_${new Date().toISOString().split('T')[0]}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
    }
</script>
@endpush
@endsection
