@extends('layouts.app')

@section('title', 'Properties Report')
@section('page-title', 'Properties Report')

@section('content')
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">Properties Report</h5>
        <div>
            <button class="btn btn-success" onclick="exportToExcel()">
                <i class="fas fa-download me-2"></i> Export
            </button>
            <button class="btn btn-primary" onclick="window.print()">
                <i class="fas fa-print me-2"></i> Print
            </button>
        </div>
    </div>

    <div class="card-body">
        <!-- Filters -->
        <div class="row mb-4">
            <div class="col-md-3">
                <form method="GET" action="{{ route('reports.properties') }}" id="filterForm">
                    <select class="form-control" name="status" onchange="this.form.submit()">
                        <option value="all" {{ request('status') == 'all' ? 'selected' : '' }}>All Status</option>
                        <option value="active" {{ request('status') == 'active' ? 'selected' : '' }}>Active</option>
                        <option value="completed" {{ request('status') == 'completed' ? 'selected' : '' }}>Completed</option>
                        <option value="cancelled" {{ request('status') == 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                    </select>
                </form>
            </div>
            <div class="col-md-6">
                <form method="GET" action="{{ route('reports.properties') }}" class="d-flex">
                    <input type="text" class="form-control me-2" name="location"
                           placeholder="Search by location..." value="{{ request('location') }}">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search"></i>
                    </button>
                    @if(request('location'))
                    <a href="{{ route('reports.properties') }}" class="btn btn-secondary ms-2">
                        <i class="fas fa-times"></i>
                    </a>
                    @endif
                </form>
            </div>
        </div>

        <!-- Summary Statistics -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card bg-primary text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">{{ $totalProperties }}</h3>
                        <small>Total Properties</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-success text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">Rs. {{ number_format($totalValue, 0) }}</h3>
                        <small>Total Value</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-info text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">Rs. {{ number_format($totalPaid, 0) }}</h3>
                        <small>Total Paid</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-warning text-white">
                    <div class="card-body text-center">
                        <h3 class="mb-0">Rs. {{ number_format($totalValue - $totalPaid, 0) }}</h3>
                        <small>Remaining Balance</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Properties Table -->
        <div class="table-responsive">
            <table class="table table-hover" id="propertiesTable">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Property Name</th>
                        <th>Location</th>
                        <th>Total Price</th>
                        <th>Paid Amount</th>
                        <th>Remaining</th>
                        <th>Installments</th>
                        <th>Customers</th>
                        <th>Status</th>
                        <th>Progress</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($properties as $property)
                    <tr>
                        <td>{{ $loop->iteration }}</td>
                        <td>
                            <strong>{{ $property->name }}</strong>
                            @if($property->description)
                            <div class="small text-muted">{{ Str::limit($property->description, 50) }}</div>
                            @endif
                        </td>
                        <td>{{ $property->location }}</td>
                        <td>
                            <strong>Rs. {{ number_format($property->total_price, 2) }}</strong>
                            <div class="small text-muted">
                                Down: Rs. {{ number_format($property->down_payment, 2) }}
                            </div>
                        </td>
                        <td>
                            <span class="text-success">
                                Rs. {{ number_format($property->installments_sum_paid_amount ?? 0, 2) }}
                            </span>
                        </td>
                        <td>
                            @php
                                $remaining = $property->total_price - ($property->installments_sum_paid_amount ?? 0);
                            @endphp
                            <span class="text-warning">
                                Rs. {{ number_format($remaining, 2) }}
                            </span>
                        </td>
                        <td>
                            {{ $property->installments_count }}
                            <div class="small text-muted">
                                Total: {{ $property->total_installments }}
                            </div>
                        </td>
                        <td>{{ $property->customers_count }}</td>
                        <td>
                            <span class="badge bg-{{ $property->status == 'active' ? 'success' : ($property->status == 'completed' ? 'primary' : 'danger') }}">
                                {{ ucfirst($property->status) }}
                            </span>
                        </td>
                        <td>
                            @php
                                $progress = $property->total_installments > 0 ?
                                    (($property->installments_count - $property->installments->where('status', 'pending')->count()) / $property->total_installments) * 100 : 0;
                            @endphp
                            <div class="progress" style="height: 10px;">
                                <div class="progress-bar bg-success" style="width: {{ $progress }}%"></div>
                            </div>
                            <small>{{ round($progress) }}%</small>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>

        @if($properties->isEmpty())
        <div class="text-center py-5">
            <i class="fas fa-building fa-3x text-muted mb-3"></i>
            <h5>No Properties Found</h5>
            <p class="text-muted">No properties match your search criteria.</p>
        </div>
        @endif
    </div>
</div>

@push('scripts')
<script>
    function exportToExcel() {
        const rows = [];
        const headers = ['Property Name', 'Location', 'Total Price', 'Down Payment', 'Paid Amount', 'Remaining', 'Total Installments', 'Completed Installments', 'Customers', 'Status', 'Progress'];

        $('#propertiesTable tbody tr').each(function() {
            const row = [];
            row.push($(this).find('td:nth-child(2) strong').text().trim());
            row.push($(this).find('td:nth-child(3)').text().trim());
            row.push($(this).find('td:nth-child(4) strong').text().replace('Rs.', '').trim());
            row.push($(this).find('td:nth-child(4) .small').text().replace('Down:', '').replace('Rs.', '').trim());
            row.push($(this).find('td:nth-child(5) span').text().replace('Rs.', '').trim());
            row.push($(this).find('td:nth-child(6) span').text().replace('Rs.', '').trim());
            row.push($(this).find('td:nth-child(7) .small').text().replace('Total:', '').trim());
            row.push($(this).find('td:nth-child(7)').text().split('\n')[0].trim());
            row.push($(this).find('td:nth-child(8)').text().trim());
            row.push($(this).find('td:nth-child(9) span').text().trim());
            row.push($(this).find('td:nth-child(10) small').text().trim());

            rows.push(row.join(','));
        });

        const csvContent = [headers.join(','), ...rows].join('\n');
        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `properties_report_${new Date().toISOString().split('T')[0]}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
    }
</script>
@endpush
@endsection
